package Data::Object;

use 5.014;

use strict;
use warnings;

use registry;
use routines;

use parent 'Exporter';

our $VERSION = '2.00'; # VERSION

our @EXPORT = qw(
  Array
  Boolean
  Box
  Code
  False
  Float
  Hash
  Number
  Regexp
  Scalar
  String
  True
  Undef
);

# FUNCTIONS

fun Array(Maybe[ArrayRef] $data) {
  $data //= [];

  require Data::Object::Array;

  return Box(Data::Object::Array->new($data));
}

fun Boolean(Maybe[Bool] $data) {
  $data //= 0;

  require Data::Object::Boolean;

  return Data::Object::Boolean->new($data)
}

fun Box(Any $data = undef) {

  require Data::Object::Box;

  return Data::Object::Box->new($data)
}

fun Code(Maybe[CodeRef] $data) {
  $data //= sub {};

  require Data::Object::Code;

  return Box(Data::Object::Code->new($data));
}

fun False() {

  require Data::Object::Boolean;

  return Data::Object::Boolean::False();
}

fun Float(Maybe[Num] $data) {
  $data //= '0.0';

  require Data::Object::Float;

  return Box(Data::Object::Float->new($data));
}

fun Hash(Maybe[HashRef] $data) {
  $data //= {};

  require Data::Object::Hash;

  return Box(Data::Object::Hash->new($data));
}

fun Number(Maybe[Num] $data) {
  $data //= 1;

  require Data::Object::Number;

  return Box(Data::Object::Number->new($data));
}

fun Regexp(Maybe[RegexpRef] $data) {
  $data //= qr/.*/;

  require Data::Object::Regexp;

  return Box(Data::Object::Regexp->new($data));
}

fun Scalar(Maybe[Ref] $data) {
  $data //= do { my $ref = ''; \$ref };

  require Data::Object::Scalar;

  return Box(Data::Object::Scalar->new($data));
}

fun String(Maybe[Str] $data) {
  $data //= '';

  require Data::Object::String;

  return Box(Data::Object::String->new($data));
}

fun True() {

  require Data::Object::Boolean;

  return Data::Object::Boolean::True();
}

fun Undef() {

  require Data::Object::Undef;

  return Box(Data::Object::Undef->new);
}

1;

=encoding utf8

=head1 NAME

Data::Object

=cut

=head1 ABSTRACT

Object-Orientation for Perl 5

=cut

=head1 SYNOPSIS

  package main;

  use Data::Object;

  my $array = Array [1..4];

  # my $iterator = $array->iterator;

  # $iterator->next; # 1

=cut

=head1 DESCRIPTION

This package automatically exports and provides constructor functions for
creating chainable data type objects from raw Perl data types.

=cut

=head1 LIBRARIES

This package uses type constraints from:

L<Data::Object::Types>

=cut

=head1 FUNCTIONS

This package implements the following functions:

=cut

=head2 array

  Array(ArrayRef $data) : InstanceOf["Data::Object::Box"]

The Array function returns a L<Data::Object::Box> which wraps a
L<Data::Object::Array> object.

=over 4

=item Array example #1

  package main;

  my $array = Array; # []

=back

=over 4

=item Array example #2

  package main;

  my $array = Array [1..4];

=back

=cut

=head2 boolean

  Boolean(Bool $data) : BooleanObject

The Boolean function returns a L<Data::Object::Boolean> object representing a
true or false value.

=over 4

=item Boolean example #1

  package main;

  my $boolean = Boolean;

=back

=over 4

=item Boolean example #2

  package main;

  my $boolean = Boolean 0;

=back

=cut

=head2 box

  Box(Any $data) : InstanceOf["Data::Object::Box"]

The Box function returns a L<Data::Object::Box> object representing a data type
object which is automatically deduced.

=over 4

=item Box example #1

  package main;

  my $box = Box;

=back

=over 4

=item Box example #2

  package main;

  my $box = Box 123;

=back

=over 4

=item Box example #3

  package main;

  my $box = Box [1..4];

=back

=over 4

=item Box example #4

  package main;

  my $box = Box {1..4};

=back

=cut

=head2 code

  Code(CodeRef $data) : InstanceOf["Data::Object::Box"]

The Code function returns a L<Data::Object::Box> which wraps a
L<Data::Object::Code> object.

=over 4

=item Code example #1

  package main;

  my $code = Code;

=back

=over 4

=item Code example #2

  package main;

  my $code = Code sub { shift };

=back

=cut

=head2 false

  False() : BooleanObject

The False function returns a L<Data::Object::Boolean> object representing a
false value.

=over 4

=item False example #1

  package main;

  my $false = False;

=back

=cut

=head2 float

  Float(Num $data) : InstanceOf["Data::Object::Box"]

The Float function returns a L<Data::Object::Box> which wraps a
L<Data::Object::Float> object.

=over 4

=item Float example #1

  package main;

  my $float = Float;

=back

=over 4

=item Float example #2

  package main;

  my $float = Float '0.0';

=back

=cut

=head2 hash

  Hash(HashRef $data) : InstanceOf["Data::Object::Box"]

The Hash function returns a L<Data::Object::Box> which wraps a
L<Data::Object::Hash> object.

=over 4

=item Hash example #1

  package main;

  my $hash = Hash;

=back

=over 4

=item Hash example #2

  package main;

  my $hash = Hash {1..4};

=back

=cut

=head2 number

  Number(Num $data) : InstanceOf["Data::Object::Box"]

The Number function returns a L<Data::Object::Box> which wraps a
L<Data::Object::Number> object.

=over 4

=item Number example #1

  package main;

  my $number = Number;

=back

=over 4

=item Number example #2

  package main;

  my $number = Number 123;

=back

=cut

=head2 regexp

  Regexp(RegexpRef $data) : InstanceOf["Data::Object::Box"]

The Regexp function returns a L<Data::Object::Box> which wraps a
L<Data::Object::Regexp> object.

=over 4

=item Regexp example #1

  package main;

  my $regexp = Regexp;

=back

=over 4

=item Regexp example #2

  package main;

  my $regexp = Regexp qr/.*/;

=back

=cut

=head2 scalar

  Scalar(Ref $data) : InstanceOf["Data::Object::Box"]

The Scalar function returns a L<Data::Object::Box> which wraps a
L<Data::Object::Scalar> object.

=over 4

=item Scalar example #1

  package main;

  my $scalar = Scalar;

=back

=over 4

=item Scalar example #2

  package main;

  my $scalar = Scalar \*main;

=back

=cut

=head2 string

  String(Str $data) : InstanceOf["Data::Object::Box"]

The String function returns a L<Data::Object::Box> which wraps a
L<Data::Object::String> object.

=over 4

=item String example #1

  package main;

  my $string = String;

=back

=over 4

=item String example #2

  package main;

  my $string = String 'abc';

=back

=cut

=head2 true

  True() : BooleanObject

The True function returns a L<Data::Object::Boolean> object representing a true
value.

=over 4

=item True example #1

  package main;

  my $true = True;

=back

=cut

=head2 undef

  Undef() : InstanceOf["Data::Object::Box"]

The Undef function returns a L<Data::Object::Undef> object representing the
I<undefined> value.

=over 4

=item Undef example #1

  package main;

  my $undef = Undef;

=back

=cut

=head1 AUTHOR

Al Newkirk, C<awncorp@cpan.org>

=head1 LICENSE

Copyright (C) 2011-2019, Al Newkirk, et al.

This is free software; you can redistribute it and/or modify it under the terms
of the The Apache License, Version 2.0, as elucidated in the L<"license
file"|https://github.com/iamalnewkirk/foobar/blob/master/LICENSE>.

=head1 PROJECT

L<Wiki|https://github.com/iamalnewkirk/foobar/wiki>

L<Project|https://github.com/iamalnewkirk/foobar>

L<Initiatives|https://github.com/iamalnewkirk/foobar/projects>

L<Milestones|https://github.com/iamalnewkirk/foobar/milestones>

L<Contributing|https://github.com/iamalnewkirk/foobar/blob/master/CONTRIBUTE.md>

L<Issues|https://github.com/iamalnewkirk/foobar/issues>

=cut
